//
//  SceneInputDataSource.m
//


#import "SceneInputDataSource.h"

@implementation SceneInputDataSource {
	NSIndexPath * _indexPath;
	id _contents;
}

- (void)setContents:(id)contents indexPath:(NSIndexPath *)indexPath originalDataSource:(DataSourceBase *)originalDataSource {
	
	self.originalDataSource = originalDataSource;
	_indexPath = indexPath;
	self.contents = contents;
	
}

- (void)setContents:(id)contents {
	_contents = contents;
	[self notifyObservers];
}

- (void)updateContents {
	if ([NSStringFromClass(self.originalDataSource.class) isEqualToString:@"SimpleDatabase"]) {
		self.contents = [self.originalDataSource datasetForItemAtIndexPath:_indexPath];
	}
	if ([NSStringFromClass(self.originalDataSource.class) isEqualToString:@"FilteredDataSource"]) {
		[self.originalDataSource performSelector:@selector(filter)];
		self.contents = [self.originalDataSource datasetForItemAtIndexPath:_indexPath];
		[self.originalDataSource refresh];
	}
}

- (void)navigateToPreviousItem {
	_indexPath = [NSIndexPath indexPathForRow:_indexPath.row-1 inSection:_indexPath.section];
	[self navigate];
}

- (void)navigateToNextItem {
	_indexPath = [NSIndexPath indexPathForRow:_indexPath.row+1 inSection:_indexPath.section];
	[self navigate];
}

- (void)navigateToItemAtIndex:(NSUInteger)index {
	_indexPath = [NSIndexPath indexPathForRow:index inSection:_indexPath.section];
	[self navigate];
}

- (void)navigate {
	if ([self.originalDataSource respondsToSelector:@selector(datasetForItemAtIndexPath:)]) {
		NSDictionary * dataset = [self.originalDataSource performSelector:@selector(datasetForItemAtIndexPath:) withObject:_indexPath];
		self.contents = dataset;
		[self notifyObservers];
	}
}

- (BOOL)canNavigateBack {
	return _indexPath.row != 0;
}

- (BOOL)canNavigateForward {
	NSUInteger itemCount = [self.originalDataSource numberOfItemsInSection:_indexPath.section];
	return _indexPath.row < itemCount - 1;
}

- (BOOL)canNavigate {
	return _indexPath != nil;
}


- (NSArray *)collectedValuesForKeyPath:(NSString *)keyPath {
	if (![keyPath hasPrefix:@"contents"])
		return nil;
	keyPath = [keyPath substringFromIndex:9];
	NSMutableArray * values = @[].mutableCopy;
	for (NSInteger i = 0, c = [self.originalDataSource numberOfItemsInSection:_indexPath.section]; i < c; i++) {
		NSDictionary * dataset = [self.originalDataSource performSelector:@selector(datasetForItemAtIndexPath:) withObject:[NSIndexPath indexPathForRow:i inSection:_indexPath.section]];
		id value = [dataset valueForKeyPath:keyPath];
		if (value != nil) [values addObject:value];
	}
	return values;
}

- (id)contents {
	return _contents;
}

- (BOOL)didLoad {
	return _contents != nil;
}

#pragma mark - DataSourceBase methods

- (NSInteger)numberOfSections {
	if (![_contents isKindOfClass:[NSArray class]]) return 0;
	return 1;
}

- (NSInteger)numberOfItemsInSection:(NSInteger)section {
	if (![_contents isKindOfClass:[NSArray class]]) return 0;
	return ((NSArray *)_contents).count;
}

- (NSDictionary *)datasetForItemAtIndexPath:(NSIndexPath *)indexPath {
	if (![_contents isKindOfClass:[NSArray class]]) return nil;
	return _contents[indexPath.row];
}


@end
